<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\InstrumentosRiasAPNCursoPreparacion;
use App\Models\clientes\HistoricoInsRiasAPNCursoPreparacion;

class InstrumentosRiasAPNCursoPreparacionController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerInstrumentos(Request $request) {
        return InstrumentosRiasAPNCursoPreparacion::on($this->cliente)
                                                ->orderBy('irc_activa', 'desc')
                                                ->orderBy('irc_pk_id', 'desc')
                                                ->get()->toArray();
    }

    public function crearInstrumento(Request $request) {
        $id = null;
        $nuevaVersion = null;

        DB::connection($this->cliente)->transaction(function() use($request, &$id, &$nuevaVersion) {
            $nuevaVersion = DB::connection($this->cliente)->select(
                'select nextval(\'"'.$this->cliente.'".seq_rias_apn_curso_preparacion\')'
            )[0]->nextval;

            $usuario = Auth::user()->id;

            $id = InstrumentosRiasAPNCursoPreparacion::on($this->cliente)->create([
                'irc_version' => $nuevaVersion,
                'irc_comentarios' => $request->irc_comentarios,
                'irc_instrumento' => json_encode($request->irc_instrumento),
                'irc_usu_creacion' => $usuario
            ])->irc_pk_id;

            HistoricoInsRiasAPNCursoPreparacion::on($this->cliente)->create([
                'irc_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hrc_accion' => 'Creación de una nueva versión del instrumento'
            ]);
        });

        return [$id, $nuevaVersion];
    }

    public function actualizarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            InstrumentosRiasAPNCursoPreparacion::on($this->cliente)->where('irc_pk_id', $id)->update([
                'irc_comentarios' => $request->irc_comentarios,
                'irc_instrumento' => $request->irc_instrumento
            ]);

            HistoricoInsRiasAPNCursoPreparacion::on($this->cliente)->create([
                'irc_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hrc_accion' => 'Actualización del instrumento'
            ]);
        });
    }

    public function activarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            $idActivo = InstrumentosRiasAPNCursoPreparacion::on($this->cliente)
                                                        ->where('irc_activa', true)
                                                        ->pluck('irc_pk_id')[0];

            // Inactivar el actual
            InstrumentosRiasAPNCursoPreparacion::on($this->cliente)->where('irc_pk_id', $idActivo)->update([
                'irc_activa' => false
            ]);

            HistoricoInsRiasAPNCursoPreparacion::on($this->cliente)->create([
                'irc_fk_id' => $idActivo,
                'usu_fk_id' => $usuario,
                'hrc_accion' => 'El instrumento pasa a estado inactivo'
            ]);

            // Activar el nuevo instrumento
            InstrumentosRiasAPNCursoPreparacion::on($this->cliente)->where('irc_pk_id', $id)->update([
                'irc_activa' => true
            ]);

            HistoricoInsRiasAPNCursoPreparacion::on($this->cliente)->create([
                'irc_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hrc_accion' => 'El instrumento pasa a estado activo'
            ]);
        });
    }

    public function obtenerHistoricoVersion(Request $request, $id) {
        return HistoricoInsRiasAPNCursoPreparacion::on($this->cliente)
                                   ->selectRaw('historico_ins_rias_apn_curso_preparacion.*, users.name')
                                   ->join('base.users', 'id', 'usu_fk_id')
                                   ->where('irc_fk_id', $id)
                                   ->orderBy('created_at', 'desc')
                                   ->get()->toArray();
    }
}
