<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;

use Firebase\JWT\JWT;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

class BoxController extends Controller
{
    private $config;
    private $token;

    public function __construct() {
        $this->config = json_decode(file_get_contents(storage_path('app/box/box_config.json')), true);
    }

    public function generarToken() {
        $claims = [
            'iss' => $this->config['boxAppSettings']['clientID'],
            'sub' => $this->config['enterpriseID'],
            'box_sub_type' => 'enterprise',
            'aud' => 'https://api.box.com/oauth2/token',
            'jti' => bin2hex(random_bytes(16)),
            'exp' => time() + 60 // Expiración del token
        ];

        $privateKeyPath = storage_path('app/box/box-private.pem');
        $passphrase = $this->config['boxAppSettings']['appAuth']['passphrase'];

        $privateKey = openssl_pkey_get_private(file_get_contents($privateKeyPath), $passphrase);

        // Firmar el token JWT
        $jwt = JWT::encode($claims, $privateKey, 'RS256', $this->config['boxAppSettings']['appAuth']['publicKeyID']);

        // Solicita el token de acceso a Box
        $client = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $client->post('https://api.box.com/oauth2/token', [
                'form_params' => [
                    'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                    'client_id' => $this->config['boxAppSettings']['clientID'],
                    'client_secret' => $this->config['boxAppSettings']['clientSecret'],
                    'assertion' => $jwt
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            $this->token = $data['access_token'];

            return ['estado' => true, 'token' => $data['access_token']];
        } catch (RequestException $e) {
            return ['estado' => false];
        }
    }

    public function listarArchivosFolder($idFolder) {
        $estado = ['estado' => false, 'data' => null];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->get('https://api.box.com/2.0/folders/'.$idFolder.'/items', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                ]
            ]);

            $estado['estado'] = true;
            $estado['data'] = json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }

    public function crearFolder($nombreFolder, $idPadre = null) {
        $estado = ['estado' => false, 'id' => null];

        $post = [
            'name' => strval($nombreFolder),
            'parent' => [
                'id' => $idPadre === null ? env('BOX_BASE_FOLDER') : $idPadre
            ]
        ];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->post('https://api.box.com/2.0/folders', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                    'Content-Type' => 'application/json'
                ],
                'json' => $post
            ]);

            $data = json_decode($response->getBody(), true);
            $estado = ['estado' => true, 'id' => $data['id']];
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }

    public function cargarArchivo($idFolder, $archivo, $nombre) {
        $estado = ['estado' => false, 'data' => null];

        $responseArchivo = $this->encontrarArchivoPorNombre($idFolder, $nombre);

        if ($responseArchivo['estado']) {
            if ($responseArchivo['id']) { // Sobrescribir el archivo existente
                $estado = $this->actualizarArchivo($responseArchivo['id'], $archivo);
            } else { // Subir nuevo archivo
                $estado = $this->cargarNuevoArchivo($idFolder, $archivo, $nombre);
            }
        }

        return $estado;
    }

    public function encontrarArchivoPorNombre($idFolder, $nombre) {
        $estado = ['estado' => false, 'id' => null];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->get('https://api.box.com/2.0/search', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                ],
                'query' => [
                    'query' => $nombre,
                    'type' => 'file',
                    'ancestor_folder_ids' => $idFolder,
                    'content_types' => 'name'
                ]
            ]);

            $results = json_decode($response->getBody()->getContents(), true);

            $estado['estado'] = true;

            foreach ($results['entries'] as $entry) {
                if ($entry['name'] === $nombre) {
                    $estado['id'] = $entry['id'];
                    break;
                }
            }
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }

    public function actualizarArchivo($idArchivo, $archivo) {
        $estado = ['estado' => false, 'data' => null];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->post('https://upload.box.com/api/2.0/files/'.$idArchivo.'/content', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                ],
                'multipart' => [
                    [
                        'name' => 'file',
                        'contents' => fopen($archivo->getPathname(), 'r'),
                        'filename' => $archivo->getClientOriginalName()
                    ]
                ]
            ]);

            $estado['estado'] = true;
            $estado['data'] = json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }

    public function cargarNuevoArchivo($idFolder, $archivo, $nombre) {
        $estado = ['estado' => false, 'data' => null];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->post('https://upload.box.com/api/2.0/files/content', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                ],
                'multipart' => [
                    [
                        'name' => 'file',
                        'contents' => fopen($archivo->getPathname(), 'r'),
                        'filename' => $nombre
                    ],
                    [
                        'name' => 'parent_id',
                        'contents' => $idFolder
                    ]
                ]
            ]);

            $estado['estado'] = true;
            $estado['data'] = json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }

    public function descargarArchivo($idArchivo) {
        $estado = ['estado' => false, 'file' => null];

        $httpClient = new Client(array('curl' => array(CURLOPT_SSL_VERIFYPEER => false)));

        try {
            $response = $httpClient->get('https://api.box.com/2.0/files/'.$idArchivo.'/content', [
                'headers' => [
                    'Authorization' => 'Bearer '.$this->token,
                ],
                'stream' => true
            ]);

            // Obtener el nombre del archivo desde las cabeceras
            $disposition = $response->getHeader('Content-Disposition')[0];
            $fileName = preg_match('/filename="(.+)"/', $disposition, $matches) ? $matches[1] : 'downloaded_file';

            // Obtener el contenido del archivo
            $body = $response->getBody();

            // Crear la respuesta para descargar el archivo
            $estado['file'] = response()->stream(function() use ($body) {
                while (!$body->eof()) {
                    echo $body->read(1024);
                }
            }, 200, [
                'Content-Type' => $response->getHeader('Content-Type')[0],
                'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
            ]);

            $estado['estado'] = true;
        } catch (RequestException $e) {
            Log::error('BOX HTTP Request failed', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        return $estado;
    }
}
